//	TorusGamesSimulation.c
//
//	© 2021 by Jeff Weeks
//	See TermsOfUse.txt

#include "TorusGames-Common.h"
#include "GeometryGamesUtilities-Common.h"
#include "GeometryGamesMatrix44.h"
#include "GeometryGamesSound.h"
#include <math.h>


//	How much can the simulation advance in one frame?
//	On OS and iOS, where the reported frame period
//	is tied to the display-link timer, MAX_FRAME_PERIOD
//	rarely comes into play.  But on Windows XP, where
//	the reported frame period is measured since the last
//	actual redraw, the frame period can be huge, because
//	redraws get suppressed while the user holds a menu open.
//	So use a fairly tight value for MAX_FRAME_PERIOD
//	so we don't lose too much of a short animation.
#define MAX_FRAME_PERIOD	0.05	//	in seconds

//	In a 3D game, how quickly should the aperture open?
#define APERTURE_VELOCITY	0.375	//	in inverse seconds

//	In a 2D game, how quickly should the ViewType zoom
//	between ViewBasicLarge and ViewBasicSmall or ViewRepeating?
#define VIEW_MAG_FACTOR_VELOCITY	2.0	//	in inverse seconds

//	In fundamental domain mode, how quickly should
//	the reset game animation spin the frame cell?
#define RESET_3D_GAME_DOMAIN_DURATION_1		0.625	//	in seconds
#define RESET_3D_GAME_DOMAIN_DURATION_2		0.125	//	in seconds
#define RESET_3D_GAME_DOMAIN_DURATION_3		0.625	//	in seconds

//	In tiling mode, how quickly should the reset game animation
//	advance and restore the near clip distance?
#define RESET_3D_GAME_TILING_DURATION_1		0.750	//	in seconds
#define RESET_3D_GAME_TILING_DURATION_2		0.125	//	in seconds
#define RESET_3D_GAME_TILING_DURATION_3		0.750	//	in seconds


static void		Update3DSimulation(ModelData *md);
static Isometry	NearestAxisAlignedOrientation(Isometry anIsometry, double *aDotProduct);
static void		UpdateAperture(ModelData *md, double aFramePeriod);
static void		UpdateViewMagFactor(ModelData *md, double aFramePeriod);
static void		Reset3DGameWithOptions(ModelData *md, ResetPurpose aResetPurpose, unsigned int aNewValue);


void ResetGame(ModelData *md)
{
	if (md->itsGameReset != NULL)
		(*md->itsGameReset)(md);
	
	ResetScrolling(md);	//	mandatory if topology has changed, optional otherwise

	md->itsChangeCount++;
}


void ResetScrolling(ModelData *md)
{
	//	Reset 2D scrolling.
	//
	//	If ChangeTopology() called us, the old Offset2D
	//	might not even be valid in the new topology,
	//	if we're passing from Topology2DKlein to Topology2DTorus.
	//
#if defined(GAME_CONTENT_FOR_SCREENSHOT) || defined(MAKE_GAME_CHOICE_ICONS)
	if (md->itsGame == Game2DCrossword || md->itsGame == Game2DWordSearch)
	{
		//	When making a screenshot or game icon, CrosswordReset() and WordSearchReset()
		//	manually adjust the scrolling.  Don't disturb the values they set.
	}
	else
	{
		md->itsOffset.itsH	= 0.0;
		md->itsOffset.itsV	= 0.0;
	}
#else
	md->itsOffset.itsH		= 0.0;
	md->itsOffset.itsV		= 0.0;
#endif

	md->itsOffset.itsFlip	= false;

#ifdef TORUS_GAMES_FOR_TALK
	if (md->itsGame == Game2DTicTacToe)
	{
		md->itsOffset.itsH = 1.0/6.0;
		md->itsOffset.itsV = 1.0/6.0;
	}
#endif
#ifdef MAKE_GAME_CHOICE_ICONS
	if (md->itsGame == Game2DApples)
	{
		md->itsOffset.itsH = 1.0/12.0;
		md->itsOffset.itsV = 1.0/12.0;
	}
#endif
#if (SHAPE_OF_SPACE_CHESSBOARD >= 1  &  SHAPE_OF_SPACE_CHESSBOARD <= 3)
	if (md->itsGame == Game2DChess)
	{
		md->itsOffset.itsH = 1.0/16.0;
		md->itsOffset.itsV = 1.0/16.0;
	}
#endif
#ifdef SHAPE_OF_SPACE_TICTACTOE
	if (md->itsGame == Game2DTicTacToe)
	{
		md->itsOffset.itsH = 1.0/6.0;
		md->itsOffset.itsV = 1.0/6.0;
	}
#endif

	//	Reset 3D scrolling.
	//
	//	If ChangeTopology() called us, the old matrix
	//	might not even be valid in the new topology,
	//	if we're passing from Topology3DKlein to an orientable topology.
	//
	Matrix44Identity(md->its3DTilingIntoFrameCell);
	
	//	Reset 3D rotation as well.
#ifdef GAME_CONTENT_FOR_SCREENSHOT
	if (md->itsGame == Game3DMaze)
	{
		//	When making a screenshot, 3D MazeReset() manually
		//	sets the orientation. Don't overwrite it.
	}
	else
	{
		md->its3DFrameCellIntoWorld = (Isometry)IDENTITY_ISOMETRY;
	}
#else
	md->its3DFrameCellIntoWorld = (Isometry)IDENTITY_ISOMETRY;
#endif
}


void SetUpGame(ModelData *md)
{
	//	Let the new game set up its internal data, allocate any memory
	//	it may need, and initialize function pointers.

	//	We assume the caller has already set up all platform-dependent items,
	//	such as fonts and a possible side window.

	switch (md->itsGame)
	{
		case GameNone:									break;

		case Game2DIntro:		     Intro2DSetUp(md);	break;
		case Game2DTicTacToe:	 TicTacToe2DSetUp(md);	break;
		case Game2DGomoku:		    Gomoku2DSetUp(md);	break;
		case Game2DMaze:		      Maze2DSetUp(md);	break;
		case Game2DCrossword:	 Crossword2DSetUp(md);	break;
		case Game2DWordSearch:	WordSearch2DSetUp(md);	break;
		case Game2DJigsaw:		    Jigsaw2DSetUp(md);	break;
		case Game2DChess:		     Chess2DSetUp(md);	break;
		case Game2DPool:		      Pool2DSetUp(md);	break;
		case Game2DApples:		    Apples2DSetUp(md);	break;

		case Game3DTicTacToe:	 TicTacToe3DSetUp(md);	break;
		case Game3DMaze:		      Maze3DSetUp(md);	break;

		default:
			GeometryGamesFatalError(u"SetUpGame() received an unknown GameType.", u"Internal Error");
			return;	//	should never occur
	}

	//	Reset the board position.
	ResetScrolling(md);	//	mandatory if topology has changed, optional otherwise

	//	Request a redraw.
	md->itsChangeCount++;
}


void ShutDownGame(ModelData *md)
{
	//	Let the old game shut down its internal data
	//	and free any memory it may have allocated.

	//	We assume all platform-dependent items, such as fonts and 
	//	a possible side window, are still present, and that
	//	the caller will release them after ShutDownGame() returns.
	//	(In practice, no current game uses fonts or a side window
	//	during shutdown, but if one ever needs them, they're there.)
	
	//	Cancel any coasting.
	md->itsCoastingStatus = CoastingNone;

	//	Clear cached sounds.
	ClearSoundCache();

	//	Let the game free its private memory,
	//	terminate any private threads,
	//	and generally clean up as it sees fit.
	//
	//	Note:  Let itsGameShutDown have the first opportunity
	//	to cancel any pending simulation, in case it also
	//	needs to free resources, stop secondary threads, etc.
	//
	if (md->itsGameShutDown != NULL)
		(*md->itsGameShutDown)(md);

	//	itsGameShutDown has probably already cancelled
	//	any pending simulation, but just to be safe
	//	call SimulationEnd() here as well.
	SimulationEnd(md);

	//	Clear function pointers.
	ClearFunctionPointers(md);
	
	//	This game is no longer available.
	md->itsGame = GameNone;
}


void SimulationBegin(
	ModelData		*md,
	SimulationType	aSimulationStatus)
{
	if (md->itsSimulationStatus != SimulationNone)
		GeometryGamesErrorMessage(	u"A new simulation is trying to begin while an old one is still active.",
						u"Internal Error in SimulationBegin()");

	md->itsSimulationStatus			= aSimulationStatus;
	md->itsSimulationElapsedTime	= 0.0;
	md->itsSimulationDeltaTime		= 0.0;
	
	//	Call SimulationUpdate(md, 0.0) to let the simulation
	//	set its parameters before the first redraw.
	SimulationUpdate(md, 0.0);
}


void SimulationEnd(ModelData *md)
{
	md->itsSimulationStatus			= SimulationNone;
	md->itsSimulationElapsedTime	= 0.0;
	md->itsSimulationDeltaTime		= 0.0;
}


bool SimulationWantsUpdates(
	ModelData	*md)
{
	//	The Chess game updates its progress indicator
	//	while the computer is choosing its next move.
	//	By contrast, the Gomoku game has no progress indicator,
	//	but it nevertheless needs to call SimulationUpdate() while
	//	itsSimulationStatus == Simulation2DGomokuChooseComputerMove
	//	so that
	//
	//		- the main thread can sleep while the secondary thread
	//			chooses the computer's next move, and
	//
	//		- the app will notice when the secondary thread
	//			has decided on a move and set itsThreadThinkingFlag = false.
	//
	//	An unfortunate side effect of this organization
	//	is that the app keeps redrawing the game board
	//	while the computer chooses its next move,
	//	but in practice it's not a problem.
	//
	return	md->itsSimulationStatus != SimulationNone
		 || md->itsCoastingStatus != CoastingNone
		 || md->its3DCurrentAperture != md->its3DDesiredAperture
		 || md->its2DViewMagFactor != md->itsDesired2DViewMagFactor;
}


void SimulationUpdate(
	ModelData	*md,
	double		aFramePeriod)	//	in seconds
{
	//	If some external delay suspends the animation for a few seconds
	//	(for example if the user holds down a menu) we'll receive
	//	a huge frame period.  To avoid a discontinuous jump,
	//	limit the frame period to MAX_FRAME_PERIOD.
	//	This limit should also have the desirable effect of slowing the animation
	//	on systems with humble graphics cards and very slow frame rates.
	if (aFramePeriod > MAX_FRAME_PERIOD)
		aFramePeriod = MAX_FRAME_PERIOD;
	
	//	Record the incremental time.
	md->itsSimulationDeltaTime = aFramePeriod;

	//	Keep track of total elapsed time.
	md->itsSimulationElapsedTime += aFramePeriod;
	
	//	Let the coasting code advance the scroll as it wishes.
	if (md->itsCoastingStatus != CoastingNone)
		CoastingMomentum(md, aFramePeriod);

	//	Let the current game respond as it wishes.
	if (md->itsGameSimulationUpdate != NULL
	 && md->itsSimulationStatus != SimulationNone)
		(*md->itsGameSimulationUpdate)(md);
	
	//	Let the 3D game-independent simulations update themselves.
	Update3DSimulation(md);

	//	Update the aperture.
	UpdateAperture(md, aFramePeriod);
	
	//	Update the small-fundamental-square magnification factor.
	UpdateViewMagFactor(md, aFramePeriod);
	
	//	The UI-specific code will need to redraw the scene.
	md->itsChangeCount++;
}

static void Update3DSimulation(
	ModelData	*md)
{
	double			s,
					t,
					theRotationHalfAngle;
	Isometry		theAnimationRotation;
	unsigned int	i;

	switch (md->itsSimulationStatus)
	{
		case SimulationNone:
			break;
		
		case Simulation3DTranslationSnapToGrid:

			if (md->itsSimulationElapsedTime > md->its3DTranslationSnapDuration)
				md->itsSimulationElapsedTime = md->its3DTranslationSnapDuration;

			t = md->itsSimulationElapsedTime / md->its3DTranslationSnapDuration;
			s = 1.0 - t;
			
			md->its3DTilingIntoFrameCell[3][0] = s * md->its3DTranslationSnapBegin[0]
											   + t * md->its3DTranslationSnapEnd  [0];
			md->its3DTilingIntoFrameCell[3][1] = s * md->its3DTranslationSnapBegin[1]
											   + t * md->its3DTranslationSnapEnd  [1];
			md->its3DTilingIntoFrameCell[3][2] = s * md->its3DTranslationSnapBegin[2]
											   + t * md->its3DTranslationSnapEnd  [2];
			md->its3DTilingIntoFrameCell[3][3] = 1.0;
			
			if (md->itsSimulationElapsedTime == md->its3DTranslationSnapDuration)
				SimulationEnd(md);

			break;

		case Simulation3DRotationSnapToAxes:

			if (md->itsSimulationElapsedTime > md->its3DRotationSnapDuration)
				md->itsSimulationElapsedTime = md->its3DRotationSnapDuration;

			InterpolateIsometries(	GeometrySpherical,
									&md->its3DRotationSnapBegin,
									&md->its3DRotationSnapEnd,
									md->itsSimulationElapsedTime / md->its3DRotationSnapDuration,
									&md->its3DFrameCellIntoWorld);
			
			if (md->itsSimulationElapsedTime == md->its3DRotationSnapDuration)
			{
				SimulationEnd(md);

				if (md->its3DRotationSnapChangeToTiling)
				{
					md->itsViewType				= ViewRepeating;
					md->its3DDesiredAperture	= 1.0;
				}
			}

			break;
		
		case Simulation3DExitTilingMode:
			if (md->its3DCurrentAperture == 0.0)
			{
				SimulationEnd(md);
				md->itsViewType = ViewBasicLarge;
			}
			break;
		
		case Simulation3DResetGameAsDomainPart1:
		
			if (md->itsSimulationElapsedTime > RESET_3D_GAME_DOMAIN_DURATION_1)
				md->itsSimulationElapsedTime = RESET_3D_GAME_DOMAIN_DURATION_1;

			t = md->itsSimulationElapsedTime / RESET_3D_GAME_DOMAIN_DURATION_1;

			theRotationHalfAngle = (0.5 * PI) * t;
			theAnimationRotation.a = cos(theRotationHalfAngle);
			theAnimationRotation.b = sin(theRotationHalfAngle) * md->its3DResetGameRotationAxis[0];
			theAnimationRotation.c = sin(theRotationHalfAngle) * md->its3DResetGameRotationAxis[1];
			theAnimationRotation.d = sin(theRotationHalfAngle) * md->its3DResetGameRotationAxis[2];
			
			ComposeIsometries(	GeometrySpherical,
								&md->its3DResetGameStartingOrientation,
								&theAnimationRotation,
								&md->its3DFrameCellIntoWorld);
			
			md->its3DResetGameScaleFactor = 1.0 - t;
			
			if (md->itsSimulationElapsedTime == RESET_3D_GAME_DOMAIN_DURATION_1)
			{
				Reset3DGameWithOptions(md, md->its3DResetPurpose, md->its3DResetNewValue);

				//	During Simulation3DResetGameAsDomainPart2,
				//	which is nothing but a wait state
				//	included for purely aesthetic reasons,
				//	the frame cell has zero size.
				md->its3DResetGameScaleFactor = 0.0;

				SimulationEnd(md);
				SimulationBegin(md, Simulation3DResetGameAsDomainPart2);
			}

			break;
		
		case Simulation3DResetGameAsDomainPart2:
		
			if (md->itsSimulationElapsedTime >= RESET_3D_GAME_DOMAIN_DURATION_2)
			{
				//	During Simulation3DResetGameAsDomainPart3,
				//	only the near wall may be transparent.
				for (i = 0; i < 6; i++)
					md->its3DResetGameOpaqueWalls[i] = (i != 4);
				
				SimulationEnd(md);
				SimulationBegin(md, Simulation3DResetGameAsDomainPart3);
			}

			break;
		
		case Simulation3DResetGameAsDomainPart3:
		
			if (md->itsSimulationElapsedTime > RESET_3D_GAME_DOMAIN_DURATION_3)
				md->itsSimulationElapsedTime = RESET_3D_GAME_DOMAIN_DURATION_3;

			t = md->itsSimulationElapsedTime / RESET_3D_GAME_DOMAIN_DURATION_3;

			//	Part 2 of the reset game animation begins
			//	a negative half turn away from the identity,
			//	and rotates around to exactly the identity.
			//
			//	its3DFrameCellIntoWorld changes discontinuously
			//	between the end of Part 1 and the beginning of Part 2,
			//	but that's OK because the frame cell has zero size at that moment.

			theRotationHalfAngle = (-0.5 * PI) * (1.0 - t);
			md->its3DFrameCellIntoWorld.a = cos(theRotationHalfAngle);
			md->its3DFrameCellIntoWorld.b = sin(theRotationHalfAngle) * md->its3DResetGameRotationAxis[0];
			md->its3DFrameCellIntoWorld.c = sin(theRotationHalfAngle) * md->its3DResetGameRotationAxis[1];
			md->its3DFrameCellIntoWorld.d = sin(theRotationHalfAngle) * md->its3DResetGameRotationAxis[2];
			
			md->its3DResetGameScaleFactor = t;
			
			if (md->itsSimulationElapsedTime == RESET_3D_GAME_DOMAIN_DURATION_3)
				SimulationEnd(md);

			break;
		
		case Simulation3DResetGameAsTilingPart1:

			if (md->itsSimulationElapsedTime > RESET_3D_GAME_TILING_DURATION_1)
				md->itsSimulationElapsedTime = RESET_3D_GAME_TILING_DURATION_1;

			md->its3DResetGameTilingFogShift = md->itsSimulationElapsedTime / RESET_3D_GAME_TILING_DURATION_1;
			
			if (md->itsSimulationElapsedTime == RESET_3D_GAME_TILING_DURATION_1)
			{
				Reset3DGameWithOptions(md, md->its3DResetPurpose, md->its3DResetNewValue);
				
				SimulationEnd(md);
				SimulationBegin(md, Simulation3DResetGameAsTilingPart2);
			}

			break;
		
		case Simulation3DResetGameAsTilingPart2:

			if (md->itsSimulationElapsedTime > RESET_3D_GAME_TILING_DURATION_2)
				md->itsSimulationElapsedTime = RESET_3D_GAME_TILING_DURATION_2;
			
			md->its3DResetGameTilingFogShift = 1.0;
			
			if (md->itsSimulationElapsedTime == RESET_3D_GAME_TILING_DURATION_2)
			{
				SimulationEnd(md);
				SimulationBegin(md, Simulation3DResetGameAsTilingPart3);
			}

			break;
		
		case Simulation3DResetGameAsTilingPart3:

			if (md->itsSimulationElapsedTime > RESET_3D_GAME_TILING_DURATION_3)
				md->itsSimulationElapsedTime = RESET_3D_GAME_TILING_DURATION_3;

			md->its3DResetGameTilingFogShift = 1.0 - md->itsSimulationElapsedTime/RESET_3D_GAME_TILING_DURATION_3;
			
			if (md->itsSimulationElapsedTime == RESET_3D_GAME_TILING_DURATION_3)
				SimulationEnd(md);

			break;
		
		default:
			//	Let the current game respond as it wishes.
			if (md->itsGameSimulationUpdate != NULL)
				(*md->itsGameSimulationUpdate)(md);
			break;
	}
}

void SnapToGrid3D(
	ModelData	*md,
	double		aMaxCoordDifference,
	double		aDuration)
{
	unsigned int	theGridSize,
					i;
	bool			theSnapIsNeeded;
	double			theCoordDifference;

	if (md->itsGame3DGridSize != NULL)
		theGridSize = (*md->itsGame3DGridSize)(md);
	else
		theGridSize = 1;
	
	theSnapIsNeeded = false;

	for (i = 0; i < 3; i++)
	{
		md->its3DTranslationSnapBegin[i]	= md->its3DTilingIntoFrameCell[3][i];
		md->its3DTranslationSnapEnd[i]		= floor(theGridSize*md->its3DTranslationSnapBegin[i] + 0.5) / (double)theGridSize;

		//	In odd-size grids, the boundary values -0.5 and +0.5 might
		//	round to the "wrong" grid point and end up outside the frame cell.
		if (md->its3DTranslationSnapEnd[i] < -0.5)
			md->its3DTranslationSnapEnd[i] += 1.0 / theGridSize;
		if (md->its3DTranslationSnapEnd[i] > +0.5)
			md->its3DTranslationSnapEnd[i] -= 1.0 / theGridSize;

		theCoordDifference = fabs(md->its3DTranslationSnapEnd[i] - md->its3DTranslationSnapBegin[i]);

		if (theCoordDifference > 0.0
		 && theCoordDifference < aMaxCoordDifference)
		{
			//	Go ahead and snap on coordinate i.
			theSnapIsNeeded = true;
		}
		else
		{
			//	Don't snap on coordinate i.
			//	If some other coordinate needs to snap,
			//	simply keep coordinate i constant.
			md->its3DTranslationSnapEnd[i] = md->its3DTranslationSnapBegin[i];
		}
	}
	md->its3DTranslationSnapBegin[3]	= 1.0;
	md->its3DTranslationSnapEnd[3]		= 1.0;

	md->its3DTranslationSnapDuration = aDuration;

	if (theSnapIsNeeded)
		SimulationBegin(md, Simulation3DTranslationSnapToGrid);
}

void SnapToAxes3D(
	ModelData	*md,
	double		aMinCosine,				//	minimum cosine of angle between
										//		current frame cell orientation ∈ Spin(3)
										//		and nearest axis-aligned placement
	double		aDuration,
	bool		aChangeToTilingFlag)	//	change to ViewRepeating when simulation ends?
{
	double	theDotProduct;	//	dot product of current frame cell orientation ∈ Spin(3)
							//		with nearest axis-aligned orientation
	bool	theSnapIsNeeded;

	md->its3DRotationSnapBegin			= md->its3DFrameCellIntoWorld;
	md->its3DRotationSnapEnd			= NearestAxisAlignedOrientation(
											md->its3DFrameCellIntoWorld,
											&theDotProduct);
	md->its3DRotationSnapDuration		= aDuration;
	md->its3DRotationSnapChangeToTiling	= aChangeToTilingFlag;

	theSnapIsNeeded = (theDotProduct > aMinCosine);

	if (theSnapIsNeeded)
		SimulationBegin(md, Simulation3DRotationSnapToAxes);
}

static Isometry NearestAxisAlignedOrientation(
	Isometry	anIsometry,
	double		*aDotProduct)	//	Records the dot product of anIsometry
								//	with the nearest axis-aligned isometry;
								//	may be NULL.
{
	double			theBestDotProduct;
	Isometry		theBestOrientation;
	double			theDotProduct;
	unsigned int	i;
	
	//	The axis-aligned orientations of a cube correspond
	//	exactly to the elements of the group Isom(cube).
	//	Because we record the cube's orientation in spin space,
	//	we must list the elements of the corresponding "binary" group
	//	which maps 2-to-1 onto Isom(cube).
	//	The group Isom(cube) == Isom(octahedron) is most commonly
	//	called the octahedral group, and its 2-fold cover
	//	is the binary octahedral group.

	//	The gcc compiler running under CodeBlocks on Windows XP
	//	considers it an error that the const variable rh
	//	"is not constant" as an array element initializer.
	//	So give it a #defined constant instead.
#define RH	ROOT_HALF
//	static const double		rh = 1.0 / ROOT2;	//	"rh" = "root half" = √½
	static const Isometry	theAxisAlignedOrientations[48] =
	{
		//	identity
		{ 1.0,  0.0,  0.0,  0.0}, {-1.0,  0.0,  0.0,  0.0},
		
		//	3-fold rotations about vertices
		{ 0.5, -0.5, -0.5, -0.5}, {-0.5,  0.5,  0.5,  0.5},
		{ 0.5, -0.5, -0.5,  0.5}, {-0.5,  0.5,  0.5, -0.5},
		{ 0.5, -0.5,  0.5, -0.5}, {-0.5,  0.5, -0.5,  0.5},
		{ 0.5, -0.5,  0.5,  0.5}, {-0.5,  0.5, -0.5, -0.5},
		{ 0.5,  0.5, -0.5, -0.5}, {-0.5, -0.5,  0.5,  0.5},
		{ 0.5,  0.5, -0.5,  0.5}, {-0.5, -0.5,  0.5, -0.5},
		{ 0.5,  0.5,  0.5, -0.5}, {-0.5, -0.5, -0.5,  0.5},
		{ 0.5,  0.5,  0.5,  0.5}, {-0.5, -0.5, -0.5, -0.5},
		
		//	2-fold rotations about edge centers
		{ 0.0,  -RH,  -RH,  0.0}, { 0.0,   RH,   RH,  0.0},
		{ 0.0,  -RH,   RH,  0.0}, { 0.0,   RH,  -RH,  0.0},
		{ 0.0,  0.0,  -RH,  -RH}, { 0.0,  0.0,   RH,   RH},
		{ 0.0,  0.0,  -RH,   RH}, { 0.0,  0.0,   RH,  -RH},
		{ 0.0,  -RH,  0.0,  -RH}, { 0.0,   RH,  0.0,   RH},
		{ 0.0,   RH,  0.0,  -RH}, { 0.0,  -RH,  0.0,   RH},
		
		//	2-fold rotations about face centers
		{ 0.0,  1.0,  0.0,  0.0}, { 0.0, -1.0,  0.0,  0.0},
		{ 0.0,  0.0,  1.0,  0.0}, { 0.0,  0.0, -1.0,  0.0},
		{ 0.0,  0.0,  0.0,  1.0}, { 0.0,  0.0,  0.0, -1.0},
		
		//	4-fold rotations about face centers
		{  RH,  -RH,  0.0,  0.0}, { -RH,   RH,  0.0,  0.0},
		{  RH,   RH,  0.0,  0.0}, { -RH,  -RH,  0.0,  0.0},
		{  RH,  0.0,  -RH,  0.0}, { -RH,  0.0,   RH,  0.0},
		{  RH,  0.0,   RH,  0.0}, { -RH,  0.0,  -RH,  0.0},
		{  RH,  0.0,  0.0,  -RH}, { -RH,  0.0,  0.0,   RH},
		{  RH,  0.0,  0.0,   RH}, { -RH,  0.0,  0.0,  -RH}
	};
	
	theBestDotProduct	= 0.0;
	theBestOrientation	= (Isometry)IDENTITY_ISOMETRY;
	
	for (i = 0; i < 48; i++)
	{
		theDotProduct = anIsometry.a * theAxisAlignedOrientations[i].a
					  + anIsometry.b * theAxisAlignedOrientations[i].b
					  + anIsometry.c * theAxisAlignedOrientations[i].c
					  + anIsometry.d * theAxisAlignedOrientations[i].d;
		
		if (theBestDotProduct < theDotProduct)
		{
			theBestDotProduct	= theDotProduct;
			theBestOrientation	= theAxisAlignedOrientations[i];
		}
	}
	
	if (aDotProduct != NULL)
		*aDotProduct = theBestDotProduct;

	return theBestOrientation;
}


static void UpdateAperture(
	ModelData	*md,
	double		aFramePeriod)
{
	//	If its3DCurrentAperture hasn't yet caught up with its3DDesiredAperture,
	//	move it along in proportion to aFramePeriod.

	if (md->its3DCurrentAperture < md->its3DDesiredAperture)
	{
		md->its3DCurrentAperture += aFramePeriod * APERTURE_VELOCITY;
		if (md->its3DCurrentAperture > md->its3DDesiredAperture)
			md->its3DCurrentAperture = md->its3DDesiredAperture;
	}

	if (md->its3DCurrentAperture > md->its3DDesiredAperture)
	{
		md->its3DCurrentAperture -= aFramePeriod * APERTURE_VELOCITY;
		if (md->its3DCurrentAperture < md->its3DDesiredAperture)
			md->its3DCurrentAperture = md->its3DDesiredAperture;
	}
}


static void UpdateViewMagFactor(
	ModelData	*md,
	double		aFramePeriod)
{
	//	If its2DViewMagFactor hasn't yet caught up with itsDesired2DViewMagFactor,
	//	move it along in proportion to aFramePeriod.

	if (md->its2DViewMagFactor < md->itsDesired2DViewMagFactor)
	{
		md->its2DViewMagFactor += aFramePeriod * VIEW_MAG_FACTOR_VELOCITY;
		if (md->its2DViewMagFactor > md->itsDesired2DViewMagFactor)
		{
			md->its2DViewMagFactor = md->itsDesired2DViewMagFactor;

			//	The animation is complete, so switch to the desired ViewType.
			md->itsViewType = md->itsDesiredViewType;
		}
	}

	if (md->its2DViewMagFactor > md->itsDesired2DViewMagFactor)
	{
		md->its2DViewMagFactor -= aFramePeriod * VIEW_MAG_FACTOR_VELOCITY;
		if (md->its2DViewMagFactor < md->itsDesired2DViewMagFactor)
		{
			md->its2DViewMagFactor = md->itsDesired2DViewMagFactor;

			//	The animation is complete, so switch to the desired ViewType.
			md->itsViewType = md->itsDesiredViewType;
		}
	}
}


bool Wall3DShouldBeTransparent(
	const     double	aWallNormalInFrameCell[4],
	/*const*/ double	aFrameCellIntoWorld[4][4],
	const     double	aScaleFactor)
{
	double	theWallCenterInFrameCell[4],
			theWallCenterInWorld[4],
			theLineOfSight[3],
			theDotProduct;
	
	static const double	theEye[3] = {0.0, 0.0, -1.0};
	
	theWallCenterInFrameCell[0]	= 0.0  +  aScaleFactor * 0.5 * aWallNormalInFrameCell[0];
	theWallCenterInFrameCell[1]	= 0.0  +  aScaleFactor * 0.5 * aWallNormalInFrameCell[1];
	theWallCenterInFrameCell[2]	= 0.0  +  aScaleFactor * 0.5 * aWallNormalInFrameCell[2];
	theWallCenterInFrameCell[3]	= 1.0;
	
	Matrix44RowVectorTimesMatrix(	theWallCenterInFrameCell,
									aFrameCellIntoWorld,
									theWallCenterInWorld);
	
	//	Let
	//
	//		c = the wall center in world coordinates
	//		e = the eye in world coordinates
	//
	//	The wall center c also happens to be a (non unit length)
	//	vector orthogonal to the wall.  So to decide whether
	//	the eye lies on the near side or the far side of the wall,
	//	compute the two dot products c·c and c·e and see which is bigger.
	//
	//	An alternative interpretation is to write c·(c - e),
	//	whose sign tells whether the line of sight from the eye
	//	to the wall's center points inward or outward.

	theLineOfSight[0] = theWallCenterInWorld[0] - theEye[0];
	theLineOfSight[1] = theWallCenterInWorld[1] - theEye[1];
	theLineOfSight[2] = theWallCenterInWorld[2] - theEye[2];
	
	theDotProduct = theWallCenterInWorld[0] * theLineOfSight[0]
				  + theWallCenterInWorld[1] * theLineOfSight[1]
				  + theWallCenterInWorld[2] * theLineOfSight[2];
	
	return theDotProduct < 0.0;
}


uint64_t GetChangeCount(ModelData *md)
{
	return md->itsChangeCount;
}


#ifdef __APPLE__
#pragma mark -
#pragma mark 3D spinning cube or fade in/out animation
#endif

void Reset3DGameWithAnimation(
	ModelData		*md,
	ResetPurpose	aResetPurpose,	//	may be ResetPlain, ResetWithNewTopology or ResetWithNewDifficultyLevel,
									//		but not ResetWithNewGame
	unsigned int	aNewValue)		//	new TopologyType or difficulty level
{
	double			theRandomAngle,
					theFrameCellIntoWorld[4][4];
	const double	*theNormalVector;
	unsigned int	i;
	
	//	Don't interrupt a pre-existing simulation.
	//
	//	Note #1:
	//	There's a temptation to cancel any pre-existing simulation,
	//	but that could be risky if the pre-existing simulation
	//	has allocated resources that it needs to free, or has
	//	spawned a secondary thread that it's waiting on.
	//
	//	Note #2:
	//	Not interrupting a pre-existing simulation works OK
	//	for the existing 3D Tic-Tac-Toe and 3D Maze,
	//	but for other game we definitely do want to interrupt
	//	some simulations, for example if the 2D Chess game
	//	is in the middle of a long computation thinking about
	//	its next move, a Reset Game or Change Game request
	//	should interrupt it.
	//
	if (md->itsSimulationStatus != SimulationNone)
		return;

	md->its3DResetPurpose	= aResetPurpose;
	md->its3DResetNewValue	= aNewValue;
	
	switch (md->itsViewType)
	{
		case ViewBasicLarge:
		
			//	Rotate the frame cell a full turn about an arbitrarily
			//	chosen axes that lies in the plane of the display,
			//	shrinking it to zero size during the first half turn
			//	and re-expanding it during the second half turn.
			//	Keep all initially opaque faces opaque, so that
			//	by the halfway point, the user can't see inside
			//	the frame cell at all.  At that moment, reset the game
			//	and set the frame cell orientation to a negative half turn
			//	about the rotation axis.  As the frame cell continues to turn,
			//	the user will see the freshly reset game, which by the end
			//	of the end of the second half turn rotates exactly
			//	into place with the identity orientation.

			md->its3DResetGameStartingOrientation = md->its3DFrameCellIntoWorld;

			theRandomAngle						= TWOPI * RandomFloat();
			md->its3DResetGameRotationAxis[0]	= cos(theRandomAngle);
			md->its3DResetGameRotationAxis[1]	= sin(theRandomAngle);
			md->its3DResetGameRotationAxis[2]	= 0.0;
			
			RealizeIsometryAs4x4MatrixInSO3(&md->its3DFrameCellIntoWorld, theFrameCellIntoWorld);
			for (i = 0; i < NUM_FRAME_WALLS; i++)
			{
				//	The wall placement matrix's third row is exactly
				//	the wall's outward-pointing normal vector.
				theNormalVector = gWallIntoFrameCellPlacements[i][2];

				md->its3DResetGameOpaqueWalls[i] = ! Wall3DShouldBeTransparent(
													theNormalVector,
													theFrameCellIntoWorld,
													1.0);
			}
			
			SimulationBegin(md, Simulation3DResetGameAsDomainPart1);

			break;
		
		case ViewRepeating:

			SimulationBegin(md, Simulation3DResetGameAsTilingPart1);

			break;
		
		case ViewBasicSmall:
			//	Should never occur.
			break;
	}
}

static void Reset3DGameWithOptions(
	ModelData		*md,
	ResetPurpose	aResetPurpose,
	unsigned int	aNewValue)	//	new TopologyType, difficulty level or ViewType
{
	switch (aResetPurpose)
	{
		case ResetPlain:
			ResetGame(md);
			break;
		
		case ResetWithNewGame:
			GeometryGamesFatalError( u"Spinning cube or fade in/out animations don’t support ResetWithNewGame,"
							u"because ResetWithNewGame also requires changes to the platform-specific UI,"
							u"such a changes to the toolbar button text.",
						u"Internal Error");
			break;
		
		case ResetWithNewTopology:
			ChangeTopology(md, (TopologyType)aNewValue);
			break;
		
		case ResetWithNewDifficultyLevel:
			ChangeDifficultyLevel(md, aNewValue);
			break;
	}
}
